/*+ TsipWindow.cpp
 *
 ******************************************************************************
 *
 *                        Trimble Navigation Limited
 *                           645 North Mary Avenue
 *                              P.O. Box 3642
 *                         Sunnyvale, CA 94088-3642
 *
 ******************************************************************************
 *
 *    Copyright  2005 Trimble Navigation Ltd.
 *    All Rights Reserved
 *
 ******************************************************************************
 *
 * Description:
 *    This file implements the CTsipWindow class.
 *            
 * Revision History:
 *    05-18-2005    Mike Priven
 *                  Written
 *
 * Notes:
 *
-*/

/*---------------------------------------------------------------------------*\
 |                         I N C L U D E   F I L E S
\*---------------------------------------------------------------------------*/
#include "stdafx.h"
#include "TsipDemo.h"
#include "TsipWindow.h"
#include "ComSelect.h"
#include "DataLogging.h"
#include ".\tsipwindow.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/*---------------------------------------------------------------------------*\
 |                  C O N S T A N T S   A N D   M A C R O S
\*---------------------------------------------------------------------------*/
#define NUM_BUTTONS 6 // Number of buttons in main GUI window

/*---------------------------------------------------------------------------*\
 |                 F O R W A R D   D E C L A R A T I O N S
\*---------------------------------------------------------------------------*/
DWORD WINAPI MonitorPort (LPVOID pParam);


/*---------------------------------------------------------------------------*\
 |                    M E T H O D   D E F I N I T I O N S
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       <constructor>

Description:    A constructor for this class. Called when a new object of this
                class is created.

Parameters:     pParent - a pointer to the parent window - the window that
                created an object of this class.

Return Value:   none
-----------------------------------------------------------------------------*/
CTsipWindow::CTsipWindow(CWnd* pParent /*=NULL*/)
    : CDialog(CTsipWindow::IDD, pParent)
{
    m_bInit = false;
    //{{AFX_DATA_INIT(CTsipWindow)
        // NOTE: the ClassWizard will add member initialization here
    //}}AFX_DATA_INIT
}

/*-----------------------------------------------------------------------------
Function:       DoDataExchange

Description:    Called by the framework to exchange and validate dialog data.
                Never call this function directly. It is called by the 
                UpdateData() member function

Parameters:     pDX - a pointer to a CDataExchange object

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::DoDataExchange(CDataExchange* pDX)
{
    CDialog::DoDataExchange(pDX);
    //{{AFX_DATA_MAP(CTsipWindow)
    DDX_Control(pDX, IDC_OUTPUT, m_edtOutput);
    //}}AFX_DATA_MAP
}

/*-----------------------------------------------------------------------------
Function:       BEGIN_MESSAGE_MAP (theClass, baseClass)

Description:    Defines a message map for this class. Matches a windows message
                with a callback function. The framework will call this function
                once the message associated with it has been received.

Parameters:     theClass - specifies the name of the class whose message map
                this is
                baseClass - specifies the name of the base class of theClass

Return Value:   none
-----------------------------------------------------------------------------*/
BEGIN_MESSAGE_MAP(CTsipWindow, CDialog)
    //{{AFX_MSG_MAP(CTsipWindow)
    ON_BN_CLICKED(IDC_COM, OnCom)
    ON_BN_CLICKED(IDC_LOG, OnLog)
    ON_BN_CLICKED(IDC_CLEAR, OnClear)
    ON_BN_CLICKED(IDC_START, OnStart)
    ON_WM_CLOSE()
    ON_WM_SIZE()
    //}}AFX_MSG_MAP
    ON_BN_CLICKED(IDC_CLOSE, OnClose)
	ON_BN_CLICKED(IDC_SEND_NMEA_PACKET, OnBnClickedSendNmeaPacket)
END_MESSAGE_MAP()

/*-----------------------------------------------------------------------------
Function:       OnCancel

Description:    The framework calls this member function when the user clicks 
                the Cancel button or presses the ESC key in a modal or 
                modeless dialog box. 

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::OnCancel()
{
    // Here, we don't do anything. That way, pressing Esc in the main GUI
    // window has no effect
}

/*-----------------------------------------------------------------------------
Function:       OnInitDialog

Description:    Initialises the dialog box. This is a standard MFC function 
                called automatically after the dialog has been created but 
                before it is displayed.

Parameters:     none

Return Value:   specifies whether the application has set the input focus to 
                one of the controls in the dialog box
-----------------------------------------------------------------------------*/
BOOL CTsipWindow::OnInitDialog() 
{
    CDialog::OnInitDialog();
    
    // Enumerate all available serial ports on the PC
    CEnumSer().EnumSerialPorts (m_Ports);

    // Attach the Trimble icon to the main window
    HICON hIcon = AfxGetApp()->LoadIcon(IDI_TRMB_BLUE);
    SetIcon (hIcon, true);
    SetIcon (hIcon, false);

    // Set the application title bar
    SetWindowText(CString(APP_NAME)+CString(" v")+CString(APP_VERSION));

    // Initialize miscellaneous class variables
    m_hMonitor   = NULL;
    m_nState     = IDLE;
    m_bFirstLine = true;
    m_bPaused    = false;

    // Retrieve the port settings from the Windows registry. If the program
    // runs for the first time, load defaults.
    m_SerialPort.SetPort(AfxGetApp()->GetProfileInt("Options","com",1));
    m_SerialPort.SetBaud(AfxGetApp()->GetProfileInt("Options","baud",COM_BAUD_9600));
    m_SerialPort.SetParity(AfxGetApp()->GetProfileInt("Options","parity",COM_PARITY_NONE));
    m_SerialPort.SetStopBits(AfxGetApp()->GetProfileInt("Options","stop",COM_STOP_BITS_1));
    m_SerialPort.SetDataBits(COM_DATA_BITS_8);

    // Initialize the data logging facility.
    m_tLog.strAsciiFile = "logAscii.txt";
    m_tLog.strRawFile  = "logRaw.log";
    m_tLog.bLogAscii    = m_tLog.bLogRaw = m_tLog.bLog = FALSE;
    m_tLog.pAsciiFile   = m_tLog.pRawFile = NULL;

    // Set the font settings for the TSIP output window
    CHARFORMAT cf;
    cf.cbSize = sizeof (CHARFORMAT);  
    cf.dwMask = CFM_FACE | CFM_SIZE; 
    cf.yHeight = 200; 
    sprintf(cf.szFaceName, "Courier New"); 
    m_edtOutput.SetDefaultCharFormat(cf); 
    m_edtOutput.SetBackgroundColor(FALSE,GetSysColor(COLOR_BTNFACE));

    // Initialize the 'Large Fonts' indicator. Certain controls must be 
    // properly resized when the system is set up for Large fonts.
    CWindowDC dc(this);
    TEXTMETRIC fontMetric;
    dc.GetTextMetrics(&fontMetric);
    m_bLargeSystemFonts = (fontMetric.tmHeight > 18) ? true : false;

	// Arrange the controls for first display
	RECT cr; 
	GetClientRect(&cr); 
	ArrangeControls(cr.right, cr.bottom); 

    // Set the global flag to indicate the program is fully initialized
    m_bInit = true;

    return TRUE;
}

/*-----------------------------------------------------------------------------
Function:       OnCom

Description:    This function is called when the COM... button is clicked.

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::OnCom() 
{
    CComSelect dlg(this);

    // Display a window to select a COM port. If the COM port was selected,
    // re-initialize the TSIP monitor thread and the new COM port.
    if ((dlg.DoModal() == IDOK) && (m_nState != IDLE))
    {
        InitMonitor();
    }
}

/*-----------------------------------------------------------------------------
Function:       OnLog

Description:    This function is called when the LOG... button is clicked.

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::OnLog() 
{
    CDataLogging dlg(this);
    dlg.DoModal();
}

/*-----------------------------------------------------------------------------
Function:       OnClear

Description:    This function is called when the Clear button is clicked.

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::OnClear() 
{
    m_edtOutput.SetWindowText("");
    m_bFirstLine = true;
}

/*-----------------------------------------------------------------------------
Function:       OnStart

Description:    This function is called when the Start button is clicked.

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::OnStart() 
{
    // The very first time the Start button is clicked, the monitor mode
    // is initialized with current COM settings. Afterwards, clicking on the
    // Start button toggles the pause/resume output functionality.
    if (m_nState == IDLE)
    {
        InitMonitor();
    }
    else
    {
        m_bPaused = !m_bPaused;
        GetDlgItem(IDC_START)->SetWindowText((m_bPaused)?"Resume":"Pause");
    }
}

/*-----------------------------------------------------------------------------
Function:       OnClose
    
Description:    This function is automatically called when a close button in
                in the upper right corner of the window is clicked on
                (the title-bar button with a cross). 

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::OnClose() 
{
    DestroyWindow();
}

/*-----------------------------------------------------------------------------
Function:       InitMonitor

Description:    Launches a thread to monitor the serial port for TSIP data.

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
bool CTsipWindow::InitMonitor()
{
    DWORD   thrdID;
    CString str;

    // If the serial port is not selected, prompt the user to select one.
    if (m_SerialPort.GetPort() <= 0) 
    {
        CComSelect dlg(this);
        dlg.DoModal();

        if (m_SerialPort.GetPort() <= 0) 
        {
            Writeln("No COM port selected!");
            return false;
        }
    }

    // If the serial port is already open, close it first.
    if (m_SerialPort.IsOpen())
    {
        m_SerialPort.Close();
    }

    // Open the serial port using currently selected settings.
    if (!m_SerialPort.Open()) 
    {
        str.Format("Could not open COM %d",m_SerialPort.GetPort());
        Writeln(str);
        return false;
    }
    else 
    {
		U8 parities[] = {'N', 'O', 'E'}; 
        str.Format("Monitoring TSIP and NMEA on COM %d @ %d baud, %d-%c-%d",
			m_SerialPort.GetPort(), m_SerialPort.GetBaud(), m_SerialPort.GetDataBits(), 
			parities[m_SerialPort.GetParity()], (m_SerialPort.GetStopBits() ? 2 : 1)
			);
        Writeln(str);

        // Save the serial port settings to the Windows registry.
        AfxGetApp()->WriteProfileInt("Options","com",m_SerialPort.GetPort());
        AfxGetApp()->WriteProfileInt("Options","parity",m_SerialPort.GetParity());
        AfxGetApp()->WriteProfileInt("Options","baud",m_SerialPort.GetBaud());
        AfxGetApp()->WriteProfileInt("Options","stop",m_SerialPort.GetStopBits());

        // If the monitor thread is already running, terminate it first.
        if (m_hMonitor)
        {
            TerminateThread(m_hMonitor, 0);
            CloseHandle(m_hMonitor);
            m_hMonitor = NULL;
        }

        // Set the program state to "monitoring TSIP".
        m_nState = TSIP;

        // Create the monitor thread to read and process TSIP packets.
        if ((m_hMonitor = CreateThread(NULL,0,
                                       (LPTHREAD_START_ROUTINE)MonitorPort,
                                       this,0,&thrdID)) == NULL)
        {
            m_SerialPort.Close();
            Writeln("Failed to create resources!");
            m_nState = IDLE;
            return false;
        }
        else
        {
            // Reset the pause/resume output to "not paused".
            m_bPaused = false;
            GetDlgItem(IDC_START)->SetWindowText("Pause");

            return true;
        }
    }
}

/*-----------------------------------------------------------------------------
Function:       MonitorPort

Description:    Runs in a thread monitoring the serial port for incoming 
                TSIP packets and launches the TSIP command processor when 
                a complete packet is received.

Parameters:     pParam - a pointer to the user specified callback data. In 
                this case, a pointer to the main dialog is passed so that 
                its routines can be used by this thread.

Return Value:   none
-----------------------------------------------------------------------------*/
DWORD WINAPI MonitorPort (LPVOID pParam)
{
    CTsipWindow   *pParent = (CTsipWindow*)pParam;
    CString       strParsedPkt;
    unsigned char ucPkt[MAX_PKT_LEN];
    int           nPktLen;

    while (pParent->m_nState == TSIP) 
    {
        // Let the CTsipParser object receive a complete TSIP packet
        // from the current serial port. Note that the ReceivePkt method of 
        // the parser object will not return until a complete TSIP packet has 
        // been received. This thread will be blocked until then.
        strParsedPkt = pParent->m_MetaParser.ReceiveAndParsePkt(&pParent->m_SerialPort,ucPkt,&nPktLen);

        // Pass both the ASCII string and the raw packet to the 
        // application so that data can be displayed in the main window and
        // logged as configured.
        pParent->ProcessPkt (strParsedPkt, ucPkt, nPktLen);
    }

    return 0;
}

/*-----------------------------------------------------------------------------
Function:       ProcessPkt

Description:    Processes the received TSIP packet.

Parameters:     str - ASCII-formatted TSIP packet data values
                ucPkt - the TSIP packet buffer
                nPktLen - size of the TSIP packet

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::ProcessPkt (CString &str, unsigned char ucPkt[], int nPktLen)
{
    // Check that the string is non-empty (i.e. the TSIP packet was 
    // successfully parsed by the CTsipParser object.
    if (str != "")
    {
        // Display the string in the main output window.
        Writeln(str);

        // Log the string to an ASCII text file if logging is enabled.
        if (m_tLog.bLog && m_tLog.bLogAscii && (m_tLog.bLogAscii != NULL))
        {
            str += "\r\n";
            m_tLog.pAsciiFile->Write(str, str.GetLength());
        }
    }

    // Check if binary logging is on.
    if (m_tLog.bLog && m_tLog.bLogRaw && (m_tLog.bLogRaw != NULL))
    {
        for (int i=0; i<nPktLen; i++)
        {
            // Log one byte at a time. This is needed so that 0x10 values
            // can be properly stuffed.
            m_tLog.pRawFile->Write(&ucPkt[i],1);

            // If the TSIP data value is 0x10, it must be stuffed with another
            // 0x10 byte. Note that 0x10 in the header and 0x10 in the end
            // of the packet are not stuffed.
            if ((ucPkt[i] == DLE) && (i>0) && (i<nPktLen-2))
            {
                m_tLog.pRawFile->Write(&ucPkt[i],1);
            }
        }
    }
}

/*-----------------------------------------------------------------------------
Function:       Writeln

Description:    Writes a new line to the output control. Scrolls the window if
                a new line is added past the end of the visible area of the
                control

Parameters:     none

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::Writeln(CString str)
{
    int nLines = 1, i = 0;

    // The application must be initialized and output non-paused.
    if (!m_bInit || m_bPaused) 
    {
        return;
    }

    // Clear the window periodically.
    if (m_edtOutput.GetLineCount() >= 5000)
    {
        OnClear();
    }

    // If the window is empty, just add the text. Otherwise, figure out
    // how many lines need to be scrolled so that the most recent output
    // is visisble in the window without manual scrolling.
    if (m_bFirstLine)
    {
        m_edtOutput.SetWindowText(str);
        m_bFirstLine = false;
    }
    else
    {
        // Count how many line break the string contains. This is needed
        // for scrolling the window so that the bottom-most data is visible
        // in the window.
        while (i<str.GetLength())
        {
            if (str[i] == '\n')
            {
                nLines++;
            }
            i++;
        }

        m_edtOutput.LineScroll(nLines);
        m_edtOutput.SetSel (m_edtOutput.GetWindowTextLength(), 
                            m_edtOutput.GetWindowTextLength());
        m_edtOutput.ReplaceSel("\r\n"+str);
        m_edtOutput.LineScroll(nLines);
    }
}

/*-----------------------------------------------------------------------------
Function:       OnSize

Description:    The framework calls this member function after the windows 
                size has changed.

Parameters:     nType - specifies the type of resizing requested
                cx - specifies the new width of the client area.
                cy - sSpecifies the new height of the client area.

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::OnSize(UINT nType, int cx, int cy) 
{
    CDialog::OnSize(nType, cx, cy);
    
    if (m_bInit)
    {
        ArrangeControls(cx, cy);

        // When the window is resized, it must be scrolled such that the
        // most recent output at the bottom is visible in the window without
        // manual scrolling.

        RECT rc; m_edtOutput.GetWindowRect(&rc);

        int nRowHeight = m_bLargeSystemFonts ? 21 : 16;
        int nLinesToScroll = 
                (rc.bottom-rc.top)/nRowHeight -
                (m_edtOutput.GetLineCount()-m_edtOutput.GetFirstVisibleLine());
    
        m_edtOutput.LineScroll(-nLinesToScroll);
    }
}

/*-----------------------------------------------------------------------------
Function:       ArrangeControls

Description:    Arranges the dialog controls on the window so that they are
                clearly visible when the window is resized.

Parameters:     nWndWidth - current width of the dialog window
                nWndHeight - current height of the dialog window

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::ArrangeControls(int nWndWidth, int nWndHeight) 
{
    int nBtnWidth = (m_bLargeSystemFonts) ? 75 : 57;
    int nTotalLen = NUM_BUTTONS*(nBtnWidth + 3) + 9;
    int nOffset = 6;

    // Resize the output window
    m_edtOutput.MoveWindow(0,0,nWndWidth,nWndHeight-42,false);

    // Reposition the COM button
    GetDlgItem(IDC_COM)->MoveWindow(nOffset,nWndHeight-33,nBtnWidth,26,false);

    // Reposition the LOG button
    nOffset += (nBtnWidth+3);
    GetDlgItem(IDC_LOG)->MoveWindow(nOffset,nWndHeight-33,nBtnWidth,26,false);

    if (nWndWidth < nTotalLen)
    {
        nWndWidth = nTotalLen;
    }

    // Adjust the offset for right-aligned buttons
    nOffset = nWndWidth-3;

    // Reposition the Close button
	nOffset -= (nBtnWidth+3);
    GetDlgItem(IDC_CLOSE)->MoveWindow(nOffset,nWndHeight-33,nBtnWidth,26,false);

    // Reposition the Start button
    nOffset -= (nBtnWidth+3);
    GetDlgItem(IDC_START)->MoveWindow(nOffset,nWndHeight-33,nBtnWidth,26,false);

    // Reposition the Clear button
    nOffset -= (nBtnWidth+3);
    GetDlgItem(IDC_CLEAR)->MoveWindow(nOffset,nWndHeight-33,nBtnWidth,26,false);
    
    // Reposition the Send NMEA Packet button
	nOffset -= (nBtnWidth+3);
    GetDlgItem(IDC_SEND_NMEA_PACKET)->MoveWindow(nOffset,nWndHeight-33,nBtnWidth,26,false);

    // Reposition the Send TSIP Packet button
	nOffset -= (nBtnWidth+3);
    GetDlgItem(IDC_SEND_TSIP_PACKET)->MoveWindow(nOffset,nWndHeight-33,nBtnWidth,26,false);

    Invalidate();
}

/*-----------------------------------------------------------------------------
Function:       ConfigureDataLog

Description:    Configures the log files.

Parameters:     bLog - indicates whether global logging is on or off.

Return Value:   none
-----------------------------------------------------------------------------*/
void CTsipWindow::ConfigureDataLog (BOOL bLog)
{
    int nFlags = CFile::modeCreate|CFile::modeWrite|CFile::shareDenyNone;

    if (bLog)
    {
        // If the global logging flag is on (Start Logging button was
        // clicked in the Data Logging window), initialize the log
        // files appropriately.
        if (m_tLog.bLogAscii && (m_tLog.pAsciiFile == NULL))
        {
            m_tLog.pAsciiFile = new CFile(m_tLog.strAsciiFile, nFlags);
        }

        if (m_tLog.bLogRaw && (m_tLog.pRawFile == NULL))
        {
            m_tLog.pRawFile = new CFile(m_tLog.strRawFile, nFlags);
        }

        m_tLog.bLog = TRUE;
    }
    else
    {
        // If the global logging flag is off, close the log files.

        m_tLog.bLog = FALSE;

        if (m_tLog.pAsciiFile != NULL)
        {
            m_tLog.pAsciiFile->Close();
            delete m_tLog.pAsciiFile;
            m_tLog.pAsciiFile = NULL;
        }

        if (m_tLog.pRawFile != NULL)
        {
            m_tLog.pRawFile->Close();
            delete m_tLog.pRawFile;
            m_tLog.pRawFile = NULL;
        }

    }
}

void CTsipWindow::OnBnClickedSendNmeaPacket()
{
	m_MetaParser.SendNmeaPkt(&m_SerialPort); 
}
